/*---------------------------------------------------------------------------*\

	FILE....: measefr.cpp
	TYPE....: C++ Program
	AUTHOR..: David Rowe
	DATE....: 4/10/01

	Test program to measure the VFRI/VFRO transfer function frequency
	response (mangitude and phase) of a VPB port.  The output is a text
	file that can be used to:

	1) Drive the TS5077 program to determine hybrid balance register 
	values for the TS5070 codecs.
	2) Drive the V4pci/Europe octave scripts to determine Return Loss
        for the device under test, e.g. for compliance testing.

	- implemented using low level functions to get precise timing between
	input and output samples, which we need to measure phase
	- calibration phase determines the freq response of the codec, which
	is then subtracted from the measured response to determine the actual
	response at the VFRO/VFRI pins of the codec.
	- for measurments, all hybrid balance registers are disabled
	
	Application 1) - Hybrid Balance
        -------------------------------

	Instructions for setting codec hybrid balance registers:

	1) Run this program, output file test.efr
	2) Run ts5077.exe
	3) Load test.efr using "Select/Enter Echo - Read"
	4) Determine balance values using "Optimisation - Quick"
	5) Test using ./measerl

	Application 2) - Return Loss
        ----------------------------

	1) run this program:
	   ./measefr /xproj/V4pci/Europe/test560.txt --offhook --octave
	2) See doco in  /xproj/V4pci/Europe for more info  

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "../src/mess.h"
#include "../src/comm.h"
#include "../src/config.h"
#include "../src/dspfifo.h"
#include "../src/timer.h"
#include "../src/timer.h"
#include "../src/wobbly.h"
#include "../src/translate.h"
#include "../src/mapdev.h"
#include "../src/vpbapi.h"
#include "../src/verbose.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define WAIT		1	// time out delay 
#define	BOARD		0	// board number
#define	N	        160	// size of processing frame
#define OFF_HK_DELAY    500     // number of ms to pause after off-hook
#define NSAM            40      // number of frames to sample
#define NCONV           20      // number of frames to wait for convergence
#define AMP             4000    // peak amplitude of test signal
#define NFREQS          15      // number of test frequencies 
#define FS              8000.0  // sampling frequency

float freqs[] = {200.0, 300.0, 400.0, 500.0, 600.0, 800.0, 1000.0, 
		 1200.0, 1500.0, 2000.0, 2500.0, 3000.0, 3200.0, 3500.0,
                 3950.0};

word offhook[] = {
	PC_LCODEC_OFFHK,
	PC_CODEC_OFFHK,
	0					// channel
};

word onhook[] = {
	PC_LCODEC_ONHK,
	PC_CODEC_ONHK,
	0					// channel
};

word compload[] = {
	PC_LSPI_LOAD,
	PC_SPI_LOAD
};

word enable[] = {
	PC_LECHO_ENABLE,
	PC_ECHO_ENABLE,
	0                     // object
};

word disable[] = {
	PC_LECHO_DISABLE,
	PC_ECHO_DISABLE,
	0                     // object
};

void set_codec_reg(int    chdev,
		   unsigned short addr, 
		   unsigned short data,  
		   Comm   *c
		  );

void play_set_hw_gain(int handle, float gain, Comm *c);
void record_set_hw_gain(int handle, float gain, Comm *c);

void measure_erl(VPBREG *v, Comm *comm, float f, float *mag, float *phase,
		 float hp);
int arg_exists(int argc, char *argv[], char *arg);
void wait_for_ring(Comm *comm);

int main(int argc, char *argv[])
{
	VPBREG *v;
	int    i, arg;
	int    ch, port, dport, hdial;
	float  av_erl;
	float  mag, phase;
	float  mag_cal[NFREQS], phase_cal[NFREQS];
	FILE   *fdata;
	float  pi = 4.0*atan(1.0);
	float  re,im;
	float  hp;

	if (argc < 2) {
		printf("usage: measefr EfrFile [options]\n");
		printf("e.g. ./measefr test1.efr\n");
		printf("--port [1..4]         VPB port to test\n");
		printf("--dial dport number   Dial out another port\n");
		printf("--offhook             Take off hook immediately\n");
		printf("--octave              Octave output file format\n");
		printf("--testoffhook port    Also take port off hook\n");
		printf("--hp coeff[0..1]      enable HP filter in TX\n");
		exit(0);
	}

	fdata = fopen(argv[1],"wt");
	assert(fdata != NULL);

	// scan command line arguments

	// option to change channel
	if ((arg = arg_exists(argc, argv, "--port")) != 0) {
	  port = atoi(argv[arg+1]);
	  assert((port >= 1) && (port <= 4));
	}
	else 
	  port = 1;
	ch = port-1;

	// initialise 

	try {
		// use high level API to dial out
		// the transition to low level will cause a momentary
		// on-hook but should work OK

		if ((arg = arg_exists(argc, argv, "--dial")) != 0) {
			dport = atoi(argv[arg+1]);
			assert((dport >= 1) && (dport <= 4));
			assert(dport != port);
			hdial = vpb_open(1, dport);
			vpb_sethook_sync(hdial, VPB_OFFHOOK);
			printf("OK, dialing %s down port %d\n", argv[arg+2],
			       dport);
			vpb_dial_sync(hdial, argv[arg+2]);
			vpb_close(hdial);
		}

		Comm comm;
		v = comm.vpbreg(BOARD);

		// configure VPB signal processing --------------------------

		config_create_object(&comm, BOARD, CODEC_AD , 0, ch, 0);
		config_create_object(&comm, BOARD, ALAW2LIN , 1, 0, 0);
		config_create_object(&comm, BOARD, ECHO     , 2, 0, 0);

		config_create_object(&comm, BOARD, FIFO_UP  , 3, 0,
				     v->szrxdf[0]);
		config_create_object(&comm, BOARD, FIFO_DOWN, 4, 1,
				     v->sztxdf[1]);

		config_create_object(&comm, BOARD, LIN2ALAW , 5, 0, 0);
		config_create_object(&comm, BOARD, CODEC_DA , 6, ch, 0);

		// delay required to make echo canc work
		config_create_object(&comm, BOARD, DELAY    , 7, 0, 0);

		config_create_wire(&comm, BOARD, 0, 1);
		config_create_wire(&comm, BOARD, 1, 2);
		config_create_wire(&comm, BOARD, 1, 3);
		config_create_wire(&comm, BOARD, 4, 5);
		config_create_wire(&comm, BOARD, 4, 5);
		config_create_wire(&comm, BOARD, 5, 6);
		config_create_wire(&comm, BOARD, 4, 7);
		config_create_wire(&comm, BOARD, 7, 2);
		
		// make sure tx & rx gains 0dB
		play_set_hw_gain(ch, 0, &comm);
		record_set_hw_gain(ch, 0, &comm);

		// start main processing loop ---------------------------

		config_run(&comm, BOARD);

		if ((arg = arg_exists(argc, argv, "--dial")) != 0) {
			// make sure dialling port remains off-hook
			offhook[2] = hdial;
			comm.PutMessageVPB(BOARD,offhook);
		}

		if ((arg = arg_exists(argc, argv, "--testoffhook")) != 0) {
			// force a 2nd test port off hook (RL testing)
			offhook[2] = atoi(argv[arg+1])-1;
			       printf("taking test port %d off hook\n",
				      offhook[2]+1);
			comm.PutMessageVPB(BOARD,offhook);
		}

		if ((arg=arg_exists(argc, argv, "--hp"))) {
		    hp = atof(argv[arg+1]);
		    printf("hp = %f\n", hp);
		}
		else
		    hp = 0;

		if ((arg = arg_exists(argc, argv, "--offhook")) != 0) {
			offhook[2] = ch;
			// dont close gyrator switch on tx port, as gyrator
			// will affect freq response
			set_codec_reg(ch, 0x0a, 0x20, &comm);  
			vpb_sleep(OFF_HK_DELAY);
		} 
		else {
			// wait for ring.........

			printf("Waiting for ring on port %d\n", port);
			wait_for_ring(&comm);
		}
		

		// first calibrate for codec gain and phase in an. loopback

		printf("Calibrating..........\n");
		set_codec_reg(ch, 0x02, 0xa2, &comm);
		for(i=0; i<NFREQS; i++) {
		  measure_erl(v, &comm, freqs[i],&mag_cal[i],&phase_cal[i],0);
		  printf("%4.0f : mag = %3.1f dB phase = %4.1f deg\n",
			 freqs[i], mag_cal[i], phase_cal[i]*180.0/pi);
		}

		// now measure freq resp

		printf("\nMeasuring..........\n");
		set_codec_reg(ch, 0x02, 0xa0, &comm);
		av_erl = 0.0;
       		if (!arg_exists(argc, argv, "--octave")) {
	       	  fprintf(fdata,"%d\r\n",NFREQS);
	      	}

		for(i=0; i<NFREQS; i++) {
		  measure_erl(v, &comm, freqs[i], &mag, &phase, hp);
		  mag -= mag_cal[i];
		  phase -= phase_cal[i];
		  printf("%4.0f : mag = %3.1f dB phase = %4.1f deg\n", 
			 freqs[i], mag, phase*180.0/pi);
		  av_erl += mag;
		  if (arg_exists(argc, argv, "--octave")) {
		    re = pow(10,mag/20)*cos(phase);
		    im = pow(10,mag/20)*sin(phase);
		    fprintf(fdata,"%f\t%f\t%f\n", freqs[i], re, im);
		  }
		  else {
		    fprintf(fdata,"%5.0f,%f,%f,1,\"*\"\r\n",freqs[i], 
			    pow(10,mag/20), phase*180/3.141);
		  }
		}

		printf("Average ERL = %3.1f dB\n",av_erl/NFREQS);

		// clean up and finish
		onhook[2] = ch;
		comm.PutMessageVPB(BOARD,onhook);
		vpb_sleep(300);

		// send clear config message -------------------------------

		if ((arg = arg_exists(argc, argv, "-dial")) != 0) {
			onhook[2] = hdial;
			comm.PutMessageVPB(BOARD,onhook);
			vpb_sleep(300);
		}
			
		config_stop(&comm, BOARD);
		config_clear(&comm, BOARD);

	}	// try ...

	catch (Wobbly w) {
		char	s[MAX_STR];
		
		w.translate(s);
		
		if (w.file[0] != 0) 
			printf("exception caught: %s, %s, line = %d\n"
			       ,s, w.file, w.line);
		else
			printf("exception caught: %s\n",s);

		printf("Press any key to exit....\n");
	}

		    fclose(fdata);
	return 0;
}

void wait_for_ring(Comm *comm) {
	char		s[MAX_STR];
	word		upmess[COMM_MAX_MESS];
	int             onhook = 1;

	verbose(1);
	while(onhook) {
		while(comm->GetMessageVPB(BOARD, upmess) == OK) {
			translate(BOARD, s, upmess);
			mprintf(s);
			if (upmess[1] == DSP_CODEC_RING) {
				offhook[2] = upmess[2];
				comm->PutMessageVPB(BOARD,offhook);
				vpb_sleep(OFF_HK_DELAY);
				onhook = 0;
			}
		}
		vpb_sleep(20);
	}
	verbose(0);
}

void measure_erl(VPBREG *v, Comm *comm, float f, float *mag, float *phase, 
		float hp) {
	int		i,j;
	char		s[MAX_STR];
	float           sam;
        short           bufrd[N], bufwr[N];
	float           echo_pwr, ref_pwr;
	word		upmess[COMM_MAX_MESS];
	float           erl;
	float           pi = 4.0*atan(1.0);
	int             ne, nr;

	float           ref_re, ref_im;
	float           echo_re, echo_im;
	float           last_sam = 0.0;
	float           samy;

	ref_pwr = echo_pwr = 0.0;
	ne = nr = 0;

	ref_re = ref_im = echo_re = echo_im = 0.0;

	for(i=0; i<NSAM; i++) {
		  
		while(comm->GetMessageVPB(BOARD, upmess) == OK) {
			translate(BOARD, s, upmess);
			mprintf(s);
		}

		// wait for buffer of echo samples

		while(v->rxdf[0]->Read((word*)bufrd, N) != OK)
			comm->CheckForAssert(BOARD);

		for(j=0; j<N; j++) { 
			sam = bufrd[j];
			if (i >= NCONV) {
				echo_re += cos(2*pi*(f/FS)*(i*N+j))*sam;
				echo_im -= sin(2*pi*(f/FS)*(i*N+j))*sam;
				echo_pwr += sam*sam;
				ne++;
			}
		}
			  
		// generate a buffer of samples and output

		for(j=0; j<N; j++) { 
			  
			sam = cos(2*pi*(f/FS)*(i*N+j));

			sam *= AMP;

			// HP filter in TX side
			samy = sam - hp*last_sam;
			last_sam = sam;

			bufwr[j] = (short)samy;
			if (i >= NCONV) {
				ref_pwr += sam*sam;
				ref_re += cos(2*pi*(f/FS)*(i*N+j))*sam;
				ref_im -= sin(2*pi*(f/FS)*(i*N+j))*sam;
				nr++;
			}
		} 

		while(v->txdf[1]->Write((word*)bufwr, N) != OK)
			comm->CheckForAssert(BOARD);
			
	}

	// calculate results
		
	erl = 10*log10(ref_pwr/echo_pwr);

	float ph_re, ph_im, ph;
	ph_re = echo_re*ref_re + echo_im*ref_im;
	ph_im = echo_im*ref_re - echo_re*ref_im;
	ph = atan2(ph_im, ph_re);

	*mag = -erl;
	*phase = ph;
}

int arg_exists(int argc, char *argv[], char *arg) {
  int i;

  for(i=0; i<argc; i++)
    if (strcmp(argv[i],arg) == 0)
      return i;

  return 0;
}






